/* Fault-tolerant barrier.  */

#ttcontext barrier
#include <stdio.h>
#include "ftlinda.h"

#define	NUM_COLS		64
#define NUM_ROWS		8
#define FAIL_ROLL       (random() % 8)

/* types */
newtype int ROW_T[NUM_COLS];
newtype ROW_T ARRAY_T[NUM_ROWS];
newtype void ARRAY; 
newtype void REGISTRY; 
newtype void WORKER_DONE;

/* function declarations */
void worker();
void monitor();
void init(ARRAY_T);
int converged(ARRAY_T, int); 
int compute(ARRAY_T, int);
static void maybe_fail(int);

/* Note: since array subscripts have not yet been implememted
 * in the AGS parsing code, we have to maintain an extra variable
 * to use in TS operations and then copy to and from outside the AGS. 
 */

@\newpage
LindaMain (argc, argv)
int argc;
char* argv [];
{
    int i, w, host, lpid, f_id, iter=1;
    ARRAY_T a;
    ROW_T r;	
    
    /* initalize the array and place it in TS */
    init(a);
    for (i=0; i<NUM_ROWS; i++) {
	(void) memcpy(r, a[i], sizeof(r));	/* copy a[i] for use in AGS */ 
	< true => out(TSmain, ARRAY, iter, i, r); >  
    }	

    /* create one monitor on each host */
    for (host=0;  host<ftl_num_hosts();  host++) {
        lpid = new_lpid();
        f_id = new_failure_id();
        ftl_create_user_thread(monitor, "monitor", host, lpid, f_id, 0, 0, 0);
    }

    /* create NUM_ROWS workers and their registry tuples */
    for (w=0; w<NUM_ROWS; w++)  {
	    host = w % ftl_num_hosts(); 	

	    /* must create registry tuple before worker! */
	    < true => out(TSmain, REGISTRY, host, w, iter); >

            lpid = new_lpid();
            ftl_create_user_thread(worker, "worker", host, lpid, w, 0, 0, 0);
    }

    /* wait until all workers are done */
    for (w=0; w<NUM_ROWS; w++)  {
        < in(TSmain, WORKER_DONE, ?w) => skip >
    }

    printf("Program %s is all done\n", argv[0]);
}

@\newpage
/* worker(id) updates a[id]) */
void
worker(int id)
{
    ARRAY_T a;
    ROW_T r;
    int i, iter, host=ftl_my_host();

    /* initialize iter and a */
    < rd(TSmain, REGISTRY, host, id,  ?iter) => skip >

    for (i=0; i<NUM_ROWS; i++) {
	< rd(TSmain, ARRAY, iter, i, ?r) => skip > /* read a[i] from TS into r */
	memcpy(a[i], r, sizeof(r));	/* copy into a[i] in memory */
    }	

    while ( !converged(a, iter) ) {

	maybe_fail(id);

	compute(a, id);		/* update a[id] in local memory */
	memcpy(r, a[i], sizeof(r)); 

	/* atomically deposit my row for next iteration & update my registry */
	< true =>
		out(TSmain, ARRAY, PLUS(iter,1), id, r);
		in(TSmain, REGISTRY, ?host, id, iter);
		out(TSmain, REGISTRY, host, id, PLUS(iter,1));
	>

	/* Barrier: wait until all workers are done with iteration iter */
	for (i=0; i<NUM_ROWS; i++) {
	    < rd(TSmain, ARRAY, PLUS(iter,1), i, ?r) => skip > 
	    memcpy(a[i], r, sizeof(r));	
	}	

	/* Garbage collection on last iteration */
	if (iter > 1) {
	    < true => in(TSmain, ARRAY, MINUS(iter,1), id, ?ROW_T); >
	}
	iter++;
    }

    < true => out(TSmain, WORKER_DONE, id); >
}
@\newpage
void
monitor(int failure_id)
{

#define ILLEGAL_WORKER		-1

    int failed_host, lpid, w, iter, my_host=ftl_my_host();

    for (;;) {

        /* wait for a failure */
        < in(TSmain, FAILURE, failure_id, ?failed_host) => skip > 

	/* try to recreate all failed workers found on this host */
        do {        
                
	    w = ILLEGAL_WORKER;

	    < inp(TSmain, REGISTRY, failed_host, ?w, ?iter) =>
		out(TSmain, REGISTRY, my_host, w, iter);
	    >

	    if (w != ILLEGAL_WORKER) {
		lpid = new_lpid();
		ftl_create_user_thread(worker, "worker", my_host, lpid, w,
								 0, 0, 0);
	    }	

        } while (w != ILLEGAL_WORKER);                
    }
#undef ILLEGAL_WORKER

}


@\newpage
/* Initialize the array somehow */
void
init(ARRAY_T a)
{
    int i,j;

    for (i=0; i<NUM_ROWS; i++)
	for (j=0; j<NUM_COLS; j++)
	    a[i][j] = 0x1000*i + j;
}

/* compute the next iteration of a[id].  Just a toy example computation ... */
int 
compute(ARRAY_T a, int id)
{
   int j;
   int above, below; 

   for (j=0; j<NUM_COLS; j++) {

	above = (id == 0 ? 0 : a[id-1][j]);
	below = (id == (NUM_ROWS-1) ? 0 : a[id+1][j]);

	a[id][j] += (above + below); 
   }	
}

/* Since this uses a toy example with no real meaning, we will simply
 * converge after a few iterations */
int 
converged(ARRAY_T a, int iterations)
{
    return (iterations <= 3 ? 0 : 1);
}

static void
maybe_fail(int id)
{
    int host = ftl_my_host();
    /* see again if we should fail our host */
    if ( (host != 0) && (FAIL_ROLL == 0) ) {
        ftl_fail_host(host);
    }
}
