/* Fault-tolerant divide and conquer worker.  Here we sum up
 * the elements of a vector to demonstrate the technique. */

#ttcontext divide
#include <stdio.h>
#include "ftlinda.h"

#define MAX_SIZE        256     /* biggest vector size */
#define MAX_ELEM        50      /* biggest element */
#define MIN_ELEM	10	/* smallest element */
#define SIZE_CUTOFF     16       
#define SMALL_ENOUGH(task)   (task.size <= SIZE_CUTOFF ? 1 : 0)
#define ILLEGAL_SIZE    -1
#define WORKERS_PER_HOST 4	/* number of workers to create on each host */

/* types */
typedef struct {
    int size;
    int elem[MAX_SIZE];
} vec;
newtype void SUBTASK; 
newtype void RESULT; 
newtype void INPROGRESS;
newtype int  SUM_T;
newtype int  SIZE_T;

/* function declarations */
void worker();
void monitor();
void init(vec *);
void part1(vec*, vec*); 
void part2(vec*, vec*); 
SUM_T sumvec(vec);
void switch_some();

@\newpage
LindaMain (argc, argv)
int argc;
char* argv [];
{
    int w, host, lpid, f_id;
    SUM_T sum, total_sum, correct_sum;
    SIZE_T size, total_size;
    vec task;
    
    /* create one monitor process on each host */
    for (host=0;  host<ftl_num_hosts();  host++) {
        lpid = new_lpid();
        f_id = new_failure_id();
        ftl_create_user_thread(monitor, "monitor", host, lpid, f_id, 0, 0, 0);
    }

    /* create WORKERS_PER_HOST workers on each host */
    for (host=0;  host<ftl_num_hosts();  host++) {
	for (w=0; w < WORKERS_PER_HOST; w++) {
            lpid = new_lpid();
            ftl_create_user_thread(worker, "worker", host, lpid, 0, 0, 0, 0);
	}
    }

    /* initalize the vector */
    init(&task);
    correct_sum = sumvec(task);        /* check answer later with this */ 

    /* deposit the task into TS */
    < true => out(TSmain, SUBTASK, task); >

    /* wait until the sums have come in from all subtasks */       
    do {

        < in(TSmain, RESULT, ?sum, ?size) => skip >
        total_sum += sum;        
        total_size += size;
        
    } while(total_size < MAX_SIZE);

    printf("The sum of the %d elements is %d\n", MAX_SIZE, total_sum);
    if (total_sum != correct_sum) 
        ftl_exit("incorrect sum", 1);
    else
        ftl_exit(NULL, 0);      /* halt the worker threads */     
}
@\newpage
void
worker()
{
    int r, host = ftl_my_host(), lpid = ftl_my_lpid();
    vec task, task1, task2;
    SUM_T sum;
    SIZE_T size;

    /* here we will put an extra LPID field in the INPROGRESS
     * tuple to ensure we withdraw our INPROGRESS tuple, not
     * another worker's from this host.
     */

    for (;;) {

        < in(TSmain, SUBTASK, ?task) =>		
            out(TSmain, INPROGRESS, task, lpid, host); >

        if (SMALL_ENOUGH(task)) {

            sum = sumvec(task);
            size = task.size;

            < in(TSmain, INPROGRESS, ?vec, lpid, host) =>
                out(TSmain, RESULT, sum, size); >
        }
        else {
            part1(&task,&task1);
            part2(&task,&task2);

            < in(TSmain, INPROGRESS, ?vec, lpid, host) =>
                    out(TSmain, SUBTASK, task1);
                    out(TSmain, SUBTASK, task2);
            >

        }
    }
}

@\newpage
void
monitor(int failure_id)
{
    int lpid=ftl_my_lpid(), failed_host, my_host=ftl_my_host();
    vec task;
    SUM_T sum;

    for (;;) {

        /* wait for a failure */
        < in(TSmain, FAILURE, failure_id, ?failed_host) => skip > 

        /* Regenerate all subtasks that were inprogress on the failed
         * host.  Note that since the AGS is not yet implemented in
         * expressions we have to test to see if the formal in the
         * inp was set.  To do this, we set task.size to an illegal
         * value; if it is still this after the inp then we know it failed.
         */

        do {        
                
            task.size = ILLEGAL_SIZE;

            < inp(TSmain, INPROGRESS, ?task, ?lpid, failed_host) =>
                out(TSmain, SUBTASK, task); >    

        } while (task.size != ILLEGAL_SIZE);                
    }
}

/* Initialize the vector randomly */
void
init(vec *task)
{
    int i, count=0;

    task->size = MAX_SIZE;
    /* seed with elements in [MIN_ELEM,MAX_ELEM) */
    for (i=0; i<MAX_SIZE; i++)  
        task->elem[i] = MIN_ELEM + (random() % (MAX_ELEM-MIN_ELEM) ); 
}

@\newpage
/* Fill the first half of t into t1 */
void
part1(vec *t, vec *t1)
{
    int i, mid = t->size / 2;

    t1->size = mid;

    for (i=0; i<mid; i++)
        t1->elem[i] = t->elem[i];    
}

/* Fill the second half of t into t2 */
void
part2(vec *t, vec *t2)
{
    int i, mid = t->size / 2;

    t2->size = t->size - mid;

    for (i=mid; i<t->size; i++)
        t2->elem[i-mid] = t->elem[i];    
}

/* Sum up the elements in task */
SUM_T
sumvec(vec task)
{
    SUM_T sum=0; int i;

    for (i=0; i<task.size; i++)
        sum += task.elem[i];

    return sum;
}
