/* 
 *		Copyright (C) 1993 by David E Bakken.  All rights reserved 
 */


/*
 * gluelib.h defines the interface for the ``glue library'' that
 * sits between the GC and FT-Linda library  and the replicated
 * tuple spaces.  This glue library may be ported to different
 * environments (e.g., Consul, SR threads package) and is thus
 * designed to be portable.  However, its structure assumes that
 * for each AGS or other FT-Linda request from the GC, a thread
 * will be created to execute in each replicated TS.  Some of
 * the functions below are called from the RTS, others from the
 * GC, and ftl_create_user_thread is called from the RTS and
 * the user's FT-Linda code.
 *
 * The code assumes that the glue library port that the RTS is
 * configured with will define all functions declared below
 * as well as struct ftl_thread_st.  It also assumes that
 * struct request_t will be defined later.
 *
 * In a few of the interfaces we define a macro fundname that calls _funcname.
 * This is because it is hard or impossible to get the GC to know about
 * request_t or formal_info_t in gluelib.h, due to circular definitions.
 * For pointers to these types we use a void pointer for the function,
 * and the macro casts the pointers to void pointers and then calls
 * the function.  The functions are defined in each gluelib port,
 * so they can actually cast these pointers back to their original
 * structures; it is just the interface definition include files that
 * are a pain.
 */

 */

/* we use ftl_thread_t as an opaque handle to a user thread. */

typedef struct ftl_thread_st *ftl_thread_t;

/*
 * ftl_create_user_thread creates a user thread and puts it on
 * the ready queue.  entry is the function that the thread is
 * to start executing with a1 through a4 as its arguments (call
 * ftl_create_user_thread with 0 for any parameters entry does 
 * not require).  entryname is a string with the name of entry,
 * e.g. "funcname".  host is the host that the the thread should
 * be created on (it is a simulated host if the gluelib implementation
 * is not distributed).  lpid is the logical PID of the thread;
 * it must be allocated with new_lpid() before calling this.
 * RTS threads will be given an LPID of 0; user threads >0. 
 *
 * a1-a4 are 4 int-sized arguments that will be passed to the thread.
 *
 * If entry returns the thread will be destroyed.  This is the normal
 * way for a user thread to complete but is an error for an RTS thread.
 *
 * Caveat: be sure that entry and anything it may call is reentrant.
 * Specifically, do not use unprotected static or global variables
 * in them, unless they're an array like
 *
 *	int my_count[MAX_THREADS];
 * 
 * and accessed like
 *
 *	my_count[MY_THREAD_ID]++;
 *
 * (MAX_THREADS and MY_THREAD_ID are defined elsewhere but are
 * available for use in the FT-Linda user code.)
 *
 * Later, if we add a user-level function to kill a user thread,
 * we will want this function to return the thread it created.
 */ 

extern void
ftl_create_user_thread(void (*entry)(),         /* FT-Linda Interface */
                       char *entryname, int host, int lpid,
                       int a1, int a2, int a3, int a4);


/*
 * ftl_scheduler lets another thread run.  If there is no other
 * ready thread then it exits.  If kill_cur_thread is TRUE then
 * it will recycle the current thread before switching to the
 * new one.  Since this implementation has no parallelism,
 * no other thread can try to use this thread structure
 * (notably its stack) until this ftl_scheduler has voluntarily
 * done a context switch and thus is done with the recycled
 * thread.
 *
 * Some implementations of the glue library (those that have to
 * do their own scheduling) will set the current thread handle
 * (static to the gluelib port) to the new thread before doing
 * a context switch to it.
 *
 * Note: ftl_scheduler does not put the current thread on any
 * list.  This is not required in FT-Linda's RTS, since an
 * opaque handle in the request structure will be set to the
 * user's thread before a request blocks.  Also, the user
 * does not need to reschedule its own threads; ftl_scheduler
 * should never be called by an FT-Linda program.
 */ 

extern void ftl_scheduler (BOOL kill_cur_thread);


/*
 * ftl_exit aborts the program with code code.  If msg is not
 * NULL then it will be printed out first.
 */

#define ftl_exit(msg,code)      _ftl_exit(msg, code, __FILE__, __LINE__)

extern void _ftl_exit(char *msg, int code, char *filename, int linenum);


/*
 * ftl_deliver_request sends request to the replicated tuple spaces.
 * request is a pointer to the request, reply_ptr is used to return to
 * the GC a pointer to a request structure with the values of the formal
 * variables filled in.  ags_num is the AGS number passed along for
 * debugging help.
 *
 * This call will block the user thread until the request has completed.
 *
 * Implementations of the glue library that do their own scheduling
 * will create an RTS thread to execute the request and also
 * will set request->r_user_thread to the current thread, so
 * the replicated TS manager can awaken the user thread when
 * the request is completed.
 *
 * Implementations of the glue library that do not send the request
 * on the network, especially those that are in one address space, 
 * may not need to use the size parameter.   
 */

#define ftl_deliver_request(request, reply, ags_num) \
        _ftl_deliver_request ( (void *) request, (void *) reply, ags_num)

extern void _ftl_deliver_request(void *request, void **reply_ptr, int ags_num);

 
/*
 * ftl_unblock will allow the user thread that submitted the
 * request to run.  It may run immediately or just be scheduled
 * for execution at a later point.
 */ 

#define ftl_unblock(request)    _ftl_unblock( (void *) request)
extern void _ftl_unblock(void *request);


/*
 * ftl_glue_init initialzes the glue library.  It should be
 * called once before any of its other functions are called.
 */ 

extern void ftl_glue_init(void);


/*
 * WARNING: this current thread pointer does not work when
 * the gluelib is not in control of scheduling threads,
 * e.g. with Consul in the x-kernel.  This is because  
 * static variables like the current thread pointer make
 * code non-reentrant.  Also, the external scheduler does
 * not know to set the current thread pointer when it 
 * runs a new thread.  Fortunately, however, such a
 * static thread pointer will ipso facto not be needed
 * in such a situation.
 */


/* ftl_my_host() returns the host number of the current thread */
extern int ftl_my_host(void);   

/* ftl_num_hosts() returns the number of hosts configured into
 * the FT-Linda system, i.e. the number of replicated TS managers,
  This is equal to NUM_HOSTS.
 */
extern int ftl_num_hosts(void);                 /* FT-Linda Interface */

extern void ftl_reschedule();	/* RTS hook; put current thread at end
				 * of ready queue and let another thread run.*/

extern int ftl_my_lpid();       /* return LPID of current thread */ 
extern char *ftl_my_name();     /* return name of current thread */  


/* ftl_my_formal_ptr returns the private formal_info_t structure
 * pointer for AGS ags_num for the current thread  */

#define ftl_my_formal_ptr(ags_num)  \
        ((formal_info_t *)_ftl_my_formal_ptr(ags_num))

extern void *_ftl_my_formal_ptr(int ags_num);


/* ftl_set_formal_ptr sets the private formal_info_t structure pointer
 * for AGS ags_num for the current thread to formal_ptr. */

#define ftl_set_formal_ptr(ags_num, formal_ptr) \
       _ftl_set_formal_ptr(ags_num, (void *) formal_ptr)

extern void _ftl_set_formal_ptr(int ags_num, void *formal_ptr);


/* ftl_my_req_ptr returns the private request_t structure pointer
 * for AGS ags_num for the current thread. */

#define ftl_my_req_ptr(ags_num) \
        ((request_t *) _ftl_my_req_ptr(ags_num))

extern void *_ftl_my_req_ptr(int ags_num);


/* ftl_set_req_ptr sets the private request_t structure pointer
 * for AGS ags_num for the current thread to req_ptr. */

#define ftl_set_req_ptr(ags_num, req_ptr) \
       _ftl_set_req_ptr(ags_num, (void *) req_ptr) 

extern void _ftl_set_req_ptr(int ags_num, void *req_ptr);



/* ftl_fail_host simulates the failure of the given host */
extern void ftl_fail_host(int host_num);


/* the gluelib should also define sr_stk_corrupted, sr_stk_overflow,
 * and sr_stk_underflow, as trivially done in SR.   We use sr_stk_underflow
 * to kill a user thread; it is an error if an RTS thread returns and
 * reaches it.
 */
 
extern void sr_stk_corrupted();
extern void sr_stk_underflow();
extern void sr_stk_overflow();
