/* Recoverable server example.  This shows how to implement two commands,
 * instead of the one shown earlier in the paper.  */

#ttcontext recoverable_server
#include <stdio.h>
#include "ftlinda.h"
#include "assert.h"

/* Anything that can be an actual in a tuple had better be cast to a type 
 * so the signatures are guaranteed to match */
#define MY_SERVICE		(int) 1
#define NUM_CLIENTS		(int) 5
#define CMD1			(int) 1	/* SQR */
#define CMD2			(int) 2	/* SUM */
#define CLIENT_LOOPS		2
#define SQR_ANS(x)		((x) * (x))
#define SUM_ANS(a,b,c)		((a) + (b) + (c))
#define INITIAL_SERVER_HOST	(int) 1
#define ILLEGAL_HOST		-1
#define ILLEGAL_SQR		-1
#define INIT_SUM		(int) 0
#define INIT_SQR		(int) 0

newtype void REQUEST;
newtype void IN_PROGRESS;
newtype void REPLY;
newtype void SERVER_REGISTRY;
newtype void SERVER_STATE;
newtype void SERVER_HANDLE;
newtype void SQR_CMD;	/* CMD1 */
newtype void SUM_CMD;	/* CMD2 */
newtype void REINCARNATE;

void server(void);
void client(int);
void monitor(int);

LindaMain (argc, argv)
int argc;
char* argv [];
{
    int f_id, host, i, server_lpid, lpid, num_hosts = ftl_num_hosts();
    ts_handle_t server_handle;

    /* For now we use a {Stable,Shared} TS for the server, instead 
     * of a {Stable,Private} one as one would normally do, since
     * passing an LPID to the create primitive isn't implemented yet. */
    server_lpid = new_lpid();
    create_TS(Stable, Shared, &server_handle);

    /* Initialize the TS handle and registry for the server.
     * We will not initialize the state, since if server_handle were
     * Private, as our solution would normally have, we could not do this.
     */
    < true =>	out(TSmain, SERVER_HANDLE, MY_SERVICE, server_handle); 
		out(TSmain, SERVER_REGISTRY, MY_SERVICE, server_lpid, INITIAL_SERVER_HOST);
    >

    /* create a monitor process on each host */
    for (host=0;  host<num_hosts;  host++) {
	lpid = new_lpid();
	f_id = new_failure_id();
	ftl_create_user_thread(monitor, "monitor", host, lpid, f_id, 0, 0, 0);
    }

    /* Create one server on host INITIAL_SERVER_HOST */
    assert(ftl_num_hosts() >= INITIAL_SERVER_HOST);
    lpid = new_lpid();
    ftl_create_user_thread(server, "server", INITIAL_SERVER_HOST, lpid, 0, 0, 0, 0);
 

    /* create some clients */
    for (i=0; i<NUM_CLIENTS; i++) {
	lpid = new_lpid();
	host = i % num_hosts;
	ftl_create_user_thread(client, "client", host, lpid, i, 0, 0, 0);
    }
    /* The LindaMain thread goes away here, but the program won't be
     * finished until all living clients are through */
}

@\newpage
/* The client invokes both services CLIENT_LOOPS times.  It also
 * tests the answers it gets, something of course a real client
 * generally would not (and often could not) do.
 */
void
client (int client_id)
{
    int i, x, a, b, c, answer, my_lpid = ftl_my_lpid();

    printf("Client %d on host %d here\n", client_id, ftl_my_host() );

    for (i=1; i<=CLIENT_LOOPS; i++) {

	/* invoke the first command */
	x=i+10;
	< true => out(TSmain, REQUEST, MY_SERVICE, my_lpid, SQR_CMD, CMD1, x); >

	/* wait for the first reply to this command */
	< in(TSmain, REPLY, MY_SERVICE, my_lpid, ?answer) => skip >
	if (answer != SQR_ANS(x))
	    ftl_exit("Client got bad sqr answer.", 1);

	/* invoke the second command */
	a=i*100; b=i*200; c=i*300;
	< true => out(TSmain, REQUEST, MY_SERVICE, my_lpid, SUM_CMD, CMD2, a, b, c); >
	
	/* wait for the first reply to this command */
	< in(TSmain, REPLY, MY_SERVICE, my_lpid, ?answer) => skip >
	if (answer != SUM_ANS(a, b, c))
	    ftl_exit("Client got bad answer.", 1);
	
    }

    printf("Client %d on host %d done\n", client_id, ftl_my_host() );

}

@\newpage
/* The server implements two different commands, both of which return
 * a simple answer.
 */
void
server()
{
    int x, a, b, c, answer, cmd, best_sqr, best_sum, client_lpid;
    ts_handle_t my_ts;

    printf("Server here on host %d\n", ftl_my_host() );

    /* read in server's TS handle */
    < rd(TSmain, SERVER_HANDLE, MY_SERVICE, ?my_ts) => skip >

    /* read in server's state.  Note that we would normally initialize
     * it like this:
     *		if ( < not rdp(my_ts, SERVER_STATE, ... ) =>
     *		    out(my_ts, SERVER_STATE, ... initial values ...); >
     * However, we cannot do this since the AGS has not yet been implemented
     * as part of an expression.  Thus, we will simulate this.
     */
    best_sqr = ILLEGAL_SQR;
    < rdp(my_ts, SERVER_STATE, MY_SERVICE, ?best_sqr, ?best_sum) => skip >
 
    if (best_sqr == ILLEGAL_SQR) {

	/* The rdp did not find a state tuple, so we will create one.
	 *
	 * Note that having these two AGSs would not work in general, 
	 * i.e. it won't have the same semantics viz. failures and concurrency
	 * as the AGS expression we would normally implement it with.
	 * We know here that it works, however, since there can be only this
	 * server executing MY_SERVICE at once. 
	 */

	best_sqr = INIT_SQR;
	best_sum = INIT_SUM;
	< true => out(my_ts, SERVER_STATE, MY_SERVICE, best_sqr, best_sum); >
	
    }
    
@\newpage
    /* loop forever  */
    for (;;) {
	/* read the next request tuple */
	< in(TSmain, REQUEST, MY_SERVICE, ?client_lpid, SQR_CMD, ?cmd, ?x) => 
		out(TSmain, IN_PROGRESS, MY_SERVICE, client_lpid, SQR_CMD, cmd, x);
	or
	   in(TSmain, REQUEST, MY_SERVICE, ?client_lpid, SUM_CMD, ?cmd, ?a, ?b, ?c) => 
		out(TSmain, IN_PROGRESS, MY_SERVICE, client_lpid, SUM_CMD, cmd, a, b, c);
	>

	/* compute the answer for the request */
	switch(cmd) {
	    case CMD1:
			answer = SQR_ANS(x);
			best_sqr = (answer > best_sqr ? answer : best_sqr);
			break;
	    case CMD2:
			answer = SUM_ANS(a,b,c);
			best_sum = (answer > best_sum ? answer : best_sum);
			break;
	    default:
			ftl_exit("Server error", 1);
	}

	/* send the reply and update state */
	switch(cmd) {
	    case CMD1:
			< in(TSmain, IN_PROGRESS, MY_SERVICE, ?int, SQR_CMD, cmd, ?int) =>
				out(TSmain, REPLY, MY_SERVICE, client_lpid, answer);
    				in(my_ts, SERVER_STATE, MY_SERVICE, ?int, ?int);
    				out(my_ts, SERVER_STATE, MY_SERVICE, best_sqr, best_sum);
			>
			break;
	    case CMD2:
			< in(TSmain, IN_PROGRESS, MY_SERVICE, ?int, SUM_CMD, cmd, ?int, ?int, ?int) =>
				out(TSmain, REPLY, MY_SERVICE, client_lpid, answer);
    				in(my_ts, SERVER_STATE, MY_SERVICE, ?int, ?int);
    				out(my_ts, SERVER_STATE, MY_SERVICE, best_sqr, best_sum);
			>
			break;
	    default:
			ftl_exit("Server error", 1);
	}
    }
    /*NOTREACHED*/
}
@\newpage
void
monitor(int failure_id)
{
    int failed_host, host, server_lpid, client_lpid, my_host = ftl_my_host(), reincarnate, x, a, b, c, cmd;
    ts_handle_t scratch_ts;

    create_TS(Volatile, Private, &scratch_ts);

    for (;;) {
	< in(TSmain, FAILURE, failure_id, ?failed_host) => skip > /* Wait for a failure */

	/* See if server MY_SERVICE was executing on the failed host.
	 * Again, we simulate an AGS in an expression here. */
	host = ILLEGAL_HOST;
	< rdp(TSmain, SERVER_REGISTRY, MY_SERVICE, ?server_lpid, ?host) => skip >
	if (host == failed_host) {

	    /* Service MY_SERVICE, which we are monitoring, has failed.
	     * Regenerate any request tuples found for MY_SERVICE.  Note
	     * that since there is only one server replica there can be
	     * at most one IN_PROGRESS tuple. */
	    < inp(TSmain, IN_PROGRESS, MY_SERVICE, ?client_lpid, SQR_CMD, ?cmd, ?x) =>
	    	out(TSmain, REQUEST, MY_SERVICE, client_lpid, SQR_CMD, cmd, x);
	    or
	      inp(TSmain, IN_PROGRESS, MY_SERVICE, ?client_lpid, SUM_CMD, ?cmd, ?a, ?b, ?c) =>
	    	out(TSmain, REQUEST, MY_SERVICE, client_lpid, SUM_CMD, cmd, a, b, c);
	    >

	    /* Attempt to start a new incarnation of the failed server.  Again,
	     * we simulate the effect of an AGS expression with the REINCARNATE tuple. */
	    < inp(TSmain, SERVER_REGISTRY, MY_SERVICE, server_lpid, failed_host) =>
		out(TSmain, SERVER_REGISTRY, MY_SERVICE, server_lpid, my_host);
		out(scratch_ts, REINCARNATE, (int) 1);
	    >

	    /* See if we did change the registry; in this case create a new * server on this host. */
	    reincarnate = 0;
	    < inp(scratch_ts, REINCARNATE, ?reincarnate) => skip >
	    if (reincarnate) 
		ftl_create_user_thread(server, "server", my_host, server_lpid, 0, 0, 0, 0);
	}
    }
}
