#include <malloc.h>

newtype void TIDS;
newtype void LOCK;
newtype void LOCK_INUSE;
newtype void VAR;
newtype void VAR_INUSE;
newtype void TS_CUR;
newtype void TS_ORIG;

/* For each transaction we keep two scratch TSs: cur_ts keeps
 * the current values of the variables, and orig_ts keeps the
 * original values of the variables plus VAR and LOCK tuples.
 * get_cur and get_orig are utility routines that fetch the
 * handles for these two TSs for a transaction.
 */
static void get_cur(tid_t, ts_handle_t *);
static void get_orig(tid_t, ts_handle_t *);

static void monitor_transactions(int failure_id);


/* init_transaction_mgr() must be called exactly once before any transaction
 * routine below is used */
void init_transaction_mgr()
{
    int lpid, f_id;
    /* create one monitor process on each host */
    for (host=0;  host<ftl_num_hosts();  host++) {
        lpid = new_lpid();
        f_id = new_failure_id();
        ftl_create_user_thread(monitor_transactions, "monitor_transactions", 
				host, lpid, f_id, 0, 0, 0);
    }

    < true => out(TSmain, TIDS, (tid_t) 1); >
}

@\newpage
/* create_var creates a variable with ID var and an initial value of val.
 * No validity check is done on val.
 */
void create_var(var_t var, val_t val)
{

    if (var == (var_t) ILLEGAL_VAR) 
	ftl_exit("create_var has a conflict with ILLEGAL_VAR", 1);

    < true =>
	out(TSmain, VAR, var, val);
	out(TSmain, LOCK, var);
    >
}

/* destroy variable var */
void destroy_var(var_t var)
{
    /* this blocks until any current transaction with var completes */
    < in(TSmain, LOCK, var) => in(TSmain, VAR, var, ?val_t); >
}

@\newpage
/* start_transaction begins a transaction involving the num_vars variables in var_list.  
 * It returns the transaction ID for this transaction or ILLEGAL_TID if the transaction 
 * could not be started (either because there were too many outstanding transactions 
 * or because a variable in var_list[]).  */
tid_t
start_transaction(var_t var_list[], int num_vars)
{
    tid_t tid; var_t *vars, var; val_t val;
    int i, my_host = ftl_my_host();
    static  int var_compare(var_t *i, var_t *j);
    ts_handle_t cur_ts, orig_ts;
    char buf[100];

    < in(TSmain, TIDS, ?tid) => out(TSmain, TIDS, PLUS(tid,1) ); >

    /* Create scratch TSs to keep a pristine copy of the variables involved
     * in this transaction as well as their current uncommittted values */
    create_TS(Volatile, Private, &cur_ts); create_TS(Volatile, Private, &orig_ts);
    < true => out(TSmain, TS_CUR, tid, cur_ts);	out(TSmain, TS_ORIG, tid, orig_ts); > 

    /* Create a safe copy of var_list and then sort it */
    vars = (var_t *) calloc(num_vars, sizeof(var_t) );
    assert(vars != (var_t) NULL);
    for (i=0; i<num_vars; i++)
	vars[i] = var_list[i];
    qsort(vars, num_vars, sizeof(var_t), var_compare);

    /* acquire all the locks for these variables in order */
    for (i=0; i < num_vars; i++) {
	var = vars[i];
	/* Move LOCK from TSmain to orig_ts and leave LOCK_INUSE in TSmain
	 * (it is used only for recovery).  Similarly for VAR; also add
	 * a copy of VAR to cur_ts.  */
	< in(TSmain, LOCK, var) =>
	    out(TSmain, LOCK_INUSE, my_host, tid, var);
	    out(orig_ts, LOCK, var);
	    in(TSmain, VAR, var, ?val);
	    out(TSmain, VAR_INUSE, my_host, tid, var, val);
	    out(orig_ts, VAR, var, val);
	    out(cur_ts, VAR, var, val);
	>
    }
    cfree(vars);
    return tid;
}
@\newpage
/* modify_var modifies var to have the value new_val.  This is assumed
 * to be called after a transaction was started with this variable.
 */
void modify_var(tid_t tid, var_t var, val_t new_val)
{
    ts_handle_t cur_ts;
    char buf[100];
 
    get_cur(tid, &cur_ts);

    sprintf(buf,"cur_ts for modify var V%d val %d tid %d", var, new_val, tid);

    < in(cur_ts, VAR, var, ?val_t) => out(cur_ts, VAR, var, new_val); >

}

/* abort aborts transaction tid */
void 
abort(tid_t tid)
{
    ts_handle_t cur_ts, orig_ts; 
    int my_host = ftl_my_host();
    char buf[100];
 
    /* Regenerate LOCK and VAR from TSmain, discard their INUSE
     * placeholders there, and remove the scratch TS handles from TS. */	
    < true =>	
	move(orig_ts, TSmain, LOCK, ?var_t);
	move(orig_ts, TSmain, VAR, ?var_t, ?val_t);
	in(TSmain, TS_CUR, tid, ?ts_handle_t);	
	in(TSmain, TS_ORIG, tid, ?ts_handle_t);	
	move(TSmain, cur_ts, VAR_INUSE, my_host, tid, ?var_t, ?val_t);
	move(TSmain, orig_ts, LOCK_INUSE, my_host, tid, ?var_t);
    >

    destroy_TS(cur_ts); destroy_TS(orig_ts);
}
@\newpage
/* commit commits transaction tid */
void
commit(tid_t tid)
{
    ts_handle_t cur_ts, orig_ts; 
    int host = ftl_my_host();
    char buf[100];

    get_cur(tid, &cur_ts);
    get_orig(tid, &orig_ts);

    /* Restore the LOCKs from this transaction from orig_ts, and move the
     *  current values of all variables involved in this transaction from 
     * cur_ts to TSmain.  Discard the VAR_INUSE and LOCK_INUSE placeholders,
     * and remove the scratch TS handles from TS. 
     */

    < true =>
	move(orig_ts, TSmain, LOCK, ?var_t);
	move(cur_ts, TSmain, VAR, ?var_t, ?val_t);
	in(TSmain, TS_CUR, tid, ?ts_handle_t);	
	in(TSmain, TS_ORIG, tid, ?ts_handle_t);	
	move(TSmain, cur_ts, VAR_INUSE, host, tid, ?var_t, ?val_t);
	move(TSmain, cur_ts, LOCK_INUSE, host, tid, ?var_t);
    >

    destroy_TS(cur_ts);
    destroy_TS(orig_ts);

    printf("Aborted transaction %d for LPID %d\n", tid, ftl_my_lpid() );
}

@\newpage
static void
monitor_transactions(int failure_id)
{
    int failed_host;
    val_t val;
    var_t var;

    for (;;) {
	/* wait for a failure */
	< in(TSmain, FAILURE, failure_id, ?failed_host) => skip >

	/* regenerate all LOCKs and VARs we find for any transactions
	 * on failed_host.  
	 */
	do {
	    var = ILLEGAL_VAR;	
	    < inp(TSmain, LOCK_INUSE, failed_host, ?tid_t, ?var) =>
		out(TSmain, LOCK, var);
		in(TSmain, VAR_INUSE, failed_host, ?tid_t, var, ?val) 
		out(TSmain, VAR, var, val);
	    >
	} while (var != ILLEGAL_VAR);

	
    }
}
@\newpage
/* print_variables will store the values into a buffer and then print, since
 * it could be scheduled out at each AGS.  That would almost certainly make
 * the output interlaced with other output, which is not very useful. */
void
print_variables(char *msg)
{
    ts_handle_t scratch_ts;
    val_t val; tid_t tid; var_t var; int host;
    char buf[1000], buf2[100];	/* Big enough ... */

    create_TS(Volatile, Private, &scratch_ts);

    /* Grab an atomic snapshot of all variables, whether in use or not. */
   < true =>
	copy(TSmain, scratch_ts, VAR, ?var_t, ?val_t);
	copy(TSmain, scratch_ts, VAR_INUSE, ?int, ?tid_t, ?var_t, ?val_t);
   >

   sprintf(buf, "Variables at %s\n", msg);
   /* Format out that snapshot, again simulating an AGS expression. */
   do {
	var = ILLEGAL_VAR;	
	< inp(scratch_ts, VAR, ?var, ?val) => skip >
	sprintf(buf2, "\tV%d=0x%x\n", var, val);
	strcat(buf, buf2);
    } while (var != ILLEGAL_VAR);
   do {
	var = ILLEGAL_VAR;	
	< inp(scratch_ts, VAR_INUSE, ?host, ?tid, ?var, ?val) => skip >
	sprintf(buf2,"\tV%d=0x%x\t(INUSE with tid %d on host %d)\n", 
		var, val, tid, host);
	strcat(buf, buf2);
    } while (var != ILLEGAL_VAR);
 
    destroy_TS(scratch_ts);
    printf(buf);
}
@\newpage
static  int
var_compare(var_t *i, var_t *j)
{
        return(*i - *j);
}


static void
get_cur(tid_t tid, ts_handle_t *handle)
{
   ts_handle_t temp;
   unsigned int old = rts_debug_value();

   < true => rd(TSmain, TS_CUR, tid, ?temp); >

   *handle = temp;
}


static void
get_orig(tid_t tid, ts_handle_t *handle)
{
   ts_handle_t temp;

   < true => rd(TSmain, TS_ORIG, tid, ?temp); >

   *handle = temp;
}
